/*
 * Copyright 2021 Alon Rashelbach
 * Apache License Version 2.0
 * See LICENSE.md for more information
 *
 * If used in academic paper, please cite:
 * Scaling Open vSwitch with a Computational Cache, USENIX NSDI'22
 *
 * https://alonrashelbach.com/libnuevomatchup
 */

#ifndef LIBNUEVOMATCHUP_H
#define LIBNUEVOMATCHUP_H

#include <stddef.h>
#include <stdint.h>
#include <stdbool.h>
#include <stdlib.h>

#ifdef __cplusplus
extern "C" {
#endif

#define LNMU_FIELD_NUM 5
#define LNMU_BATCH_SIZE 32
#define LNMU_RQRMI_BATCH_SIZE 8

struct lnmu_iset_match;
struct lnmu_rqrmi;
struct lnmu_rqrmi64;
struct lnmu_rangearr;

/**
 * @brief A user-defined function for manual flow validation.
 * @param iset_match A candidate flow to match
 * @param args User defined arguments that propagate through the validation
 * process.
 * @return Should return 1 iff the match is successful.
 */
typedef int(*lnmu_validation_lookup_func_t)
           (const struct lnmu_iset_match * iset_match, void * args);

struct lnmu_trainer_configuration {
    /* Number of samples per session */
    uint32_t samples;
    /* The maximum error threshold for an iset */
    uint16_t error_threshold;
    /* The maximum collision between rules allowed in an iset */
    uint8_t max_collision;
    /* Maximum RQ-RMI training sessions */
    uint8_t max_sessions;
    /* The minimal coverage (percent) for a new iset */
    float minimal_coverage;
    /* Invalidate RQ-RMI models if their error threshold is not reached */
    bool allow_failure;
    /* Use SIMD instructions when training (experimental) */
    bool use_batching;
    /* Use hybrid training (experimental) */
    bool use_hybrid;
};

/* Default trainer configuration */
static struct lnmu_trainer_configuration __lnmu_default_configuration =
    (struct lnmu_trainer_configuration){4000,128,16,6,0.25,false,false,false};
#define LNMU_DEFAULT_CONFIGURATION (&__lnmu_default_configuration);

/* A single 32-bit field */
struct lnmu_field {
    uint32_t low;
    uint32_t high;
};

/* flow flow */
struct lnmu_flow {
    struct lnmu_field fields[LNMU_FIELD_NUM];
    uint32_t priority; /* Higher is more prioritized, 0 is invalid */
    void *match;       /* User defined match; returned by the classifier */
    void *args;        /* User defined arguments*/
};

/* Rule inclusion policy */
enum lnmu_inclusion_policy {
    LNMU_INCLUSION_NONE = 0,
    LNMU_INCLUSION_ALLOW_IN_ISET = 1,
    LNMU_INCLUSION_ALLOW_IN_REMAINDER = 2
};

/* Interface for training NuevoMatchUp */
struct lnmu_trainer {
    struct lnmu_trainer_impl *data;
    char *error;
};

/* Additional information for iset rules */
struct lnmu_iset_match {
    void *match; /* User defined match, set automatically from "lnmu_flow" */
    void *args;  /* User defined args, set automatically from "lnmu_flow" */
    void *data;  /* Additional user arguments for the match. */
};

/* iset interface for lnmu */
struct lnmu_iset {
    size_t num_of_entries; /* iNumber of iset buckets */
    size_t num_of_rules;   /* Total nubmer of rules */

    uint32_t *value_db;
    uint32_t *validation_db;
    struct lnmu_iset_match *match_db;
    int *entry_rules;

    void *args;          /* Additional data for iset defined by the user */

    uint32_t field_idx;  /* flow field index */
    uint32_t collisions; /* Number of buckets */

    int iterations;
    int row_size;
    int hi_values;
};

/* NuevoMatchUp interface */
struct lnmu_nuevomatchup {
    size_t num_of_isets;       /* Number of isets */
    struct lnmu_rqrmi **rqrmi; /* rqrmi model array */
    struct lnmu_iset **iset;   /* iset aray */
    int version;               /* The classifier's unique version */
};

/**
 * @brief Initiate NuevoMatchUp trainier object "lnm" using the configuration
 * from "config".
 * @param config Use LNMU_DEFAULT_CONFIGURATION for the default configuration.
 * @return 0 on success. Use lnmu_get_error to see error message.
 */
int lnmu_init(struct lnmu_trainer *lnm,
              struct lnmu_trainer_configuration *config);

/**
 * @brief Destroies a trainer object
 */
void lnmu_destroy(struct lnmu_trainer *lnm);

/**
 * @brief Returns a pointer to the last error of "lnm"
 */
const char * lnmu_get_error(struct lnmu_trainer *lnm);

/**
 * @brief Inserts a flow into the trainer
 * @param lnm Trainer object
 * @param flow Pointer to flow to insert
 * @param remainder_p Pointer to remainder object, can be NULL.
 * @param unique_id Flow unique id (enforced by the user).
 * @return 0 on success. Use lnmu_get_error to see error message.
 */
int lnmu_insert(struct lnmu_trainer *lnm,
                struct lnmu_flow *flow,
                void *remainder_p,
                uint32_t *unique_id);

/**
 * @brief Removes a flow from the trainder using its unique id.
 * @param match If not NULL, *match is set to the match object associated
 * with the flow.
 * @return 0 on success. Use lnmu_get_error to see error message.
 */
int lnmu_remove(struct lnmu_trainer *lnm,
                uint32_t unique_id,
                void **match);

/**
 * @brief Updates the priority and/or inclusion policy of a flow.
 * @param lnm Trainer object
 * @param unique_id flow unique id
 * @param priority The flow's new priority (higher is better,
 * zero for invalid).
 * @param inclusion The flow inclusion policy: 1 - allow only in isets,
 * 2 - allow only in remainder, 3 - allow in both.
 */
void lnmu_update_rule(struct lnmu_trainer *lnm,
                      uint32_t unique_id,
                      uint32_t priority,
                      int inclusion);

/**
 * @brief Returns NuevoMatchUp coverage value.
 */
double lnmu_coverage(struct lnmu_trainer *lnm);

/**
 * @brief Returns NuevoMatchUp number of isets.
 */
size_t lnmu_iset_num(struct lnmu_trainer *lnm);

/**
 * @brief Returns the number of rules in isets.
 */
size_t lnmu_num_of_iset_rules(struct lnmu_trainer *lnm);

/**
 * @brief Returns the number of rules in remainder.
 */
size_t lnmu_num_of_remainder_rules(struct lnmu_trainer *lnm);

/**
 * @brief Populates "ids" with all the remainder flows' IDs. "ids" must be
 * large enough to contain the data. Returns 0 on success.
 */
int lnmu_get_remainder_rules_ids(struct lnmu_trainer *lnm, uint32_t *ids);

/**
 * @brief Set "priority" and "match" by a flow's "id".
 * @return 0 on success.
 */
int lnmu_get_flow_by_id(struct lnmu_trainer *lnm,
                        uint32_t id,
                        uint32_t *priority,
                        void **match);

/**
 * @brief Returns the number of total rules in NuevoMatchUp.
 */
size_t lnmu_num_of_rules(struct lnmu_trainer *lnm);

/**
 * @brief Returns true iff "lnm" contains a flow with id "id".
 */
bool lnmu_contains(struct lnmu_trainer *lnm, size_t id);

/**
 * @brief Train a new version of NuevoMatchUp classifier.
 * @param lnm NuevoMatchUp trainer object.
 * @param bytes If not NULL, returns the number of bytes used by the classifier
 * @return 0 on success. Use lnmu_get_error to see error message.
 */
int lnmu_update(struct lnmu_trainer *lnm, size_t *bytes);

/**
 * @brief Get the last trained classifier as "struct nuevomatchup".
 * @param lnm Trainer object
 * @param nm A pointer to NuevoMatchUp classifier to be updated.
 * @return 0 on success. Use lnmu_get_error to see error message.
 */
int lnmu_export(struct lnmu_trainer *lnm, struct lnmu_nuevomatchup *nm);

/**
 * @brief Returns the subset index for flow with id "id".
 * @note The value -1 indicates the flow is in the remainder. A non-negative
 * value indices iset index.
 */
int lnmu_get_subset_idx(struct lnmu_trainer *lnm, size_t id);

/**
 * @brief Delete all flows from the trainer.
 * @return 0 on success. Use lnmu_get_error to see error message.
 */
int lnmu_clear(struct lnmu_trainer *lnm);

/**
 * @brief Returns the user-defined arguments of all rules held by "lnm".
 * @param lnm Trainer object
 * @param rules An array of pointers, allocated by the user. Will be filled
 * by this with pointers to the rules' arguments. The array must contain
 * at least "iset_size" + "remainder_size" entries.
 * @param iset_size Number of iset rules.
 * @param remainder_size Number of remainder rules.
 */
void lnmu_get_rule_args(struct lnmu_trainer *lnm,
                        void **rules,
                        size_t iset_size,
                        size_t remainder_size);

/**
 * @brief Creates a new remainder classifier (tuplemrege) and returns it.
 */
void* lnmu_remainder_init(struct lnmu_trainer *lnm);

/**
 * @brief Destroy the remainder classifier.
 */
void lnmu_remainder_destroy(void *remainder_p);

/**
 * @brief Insert the flow with "id" to the remainder.
 * @param lnm Trainer object
 * @param remainder_p Remainder classifier
 * @param rule_id The flow's unique ID.
 * @return 0 on success. Use lnmu_get_error to see error message.
 */
int lnmu_remainder_insert(struct lnmu_trainer *lnm,
                          void *remainder_p,
                          uint32_t rule_id);

/**
 * @brief Removes a flow with the given ID from the remainder.
 * @param lnm Trainer object
 * @param remainder_p Remainder classifier
 * @param rule_id The flow's unique ID.
 * @return 0 on success. Use lnmu_get_error to see error message.
 */
int lnmu_remainder_remove(struct lnmu_trainer *lnm,
                          void *remainder_p,
                          uint32_t rule_id);

/**
 * @brief Mark a flow with the given ID as invalid in the remainder classifer
 * @param lnm Trainer object
 * @param remainder_p Remainder classifier
 * @param rule_id The flow's unique ID.
 * @return 0 on success. Use lnmu_get_error to see error message.
 */
int lnmu_remainder_mark_delete(struct lnmu_trainer *lnm,
                               void *remainder_p,
                               uint32_t rule_id);

/**
 * @brief Build the remainder classifier.
 * @param lnm Trainer object
 * @param remainder_p Remainder classifier
 * @return 0 on success. Use lnmu_get_error to see error message.
 */
int lnmu_remainder_build(struct lnmu_trainer *lnm, void *remainder_p);

/**
 * @brief Returns 1 iff a flow with id "rule_id" exists within the reaminder
 * classifier
 * @param lnm Trainer object
 * @param remainder_p Remainder classifier
 * @param rule_id The flow's unique ID. * @param lnm Trainer object
 * @param remainder_p Remainder classifier
 * @param rule_id The flow's unique ID.
 * @return
 */
int lnmu_remainder_contains(struct lnmu_trainer *lnm,
                            void *remainder_p,
                            uint32_t rule_id);

/**
 * @brief Perform classification using the remainder classifier.
 * @param lnm Trainer object
 * @param remainder_p Remainder classifier
 * @param header_fields Array of 5 32-bit values of a single packet.
 * @param id Set by this, the unique ID of the matching flow.
 * @param priority Set by this, the priority of the matching flow.
 * @param result Set by this, a pointer to the user-defined "match" of the
 * matching flow.
 * @param args Set by this, a pointer to the user-defined "args" of the 
 * matching flow.
 * @return 0 on success. Use lnmu_get_error to see error message.
 */
int lnmu_remainder_classify(struct lnmu_trainer *lnm,
                            void *remainder_p,
                            uint32_t *header_fields,
                            uint32_t *id,
                            int *priority,
                            void **result,
                            void **args);

/**
 * @brief Returns 1 iff "rqrmi" points to a valid RQRMI model.
 */
int lnmu_rqrmi_valid(const struct lnmu_rqrmi *rqrmi);

/**
 * @brief Perfom "rqrmi" inference on "inputs". Sets "outputs" and "errors".
 * @param rqrmi The RQRMI model
 * @param inputs An array of size LNMU_RQRMI_BATCH_SIZE.
 * @param outputs An array of size LNMU_RQRMI_BATCH_SIZE. Set by this.
 * @param errors An array of size LNMU_RQRMI_BATC_SIZE. Set by this.
 * @param batch_size Set batch size.
 */
void lnmu_rqrmi_inference(const struct lnmu_rqrmi *rqrmi,
                          const float *inputs,
                          float *outputs,
                          int *errors,
                          const int batch_size);

/**
 * @brief Perform iSet validation on "input".
 * @param iset The iSet
 * @param input An array of LNMU_FIELD_NUM integers that correspond to a
 * single input
 * @param index The index of the current input within the iSet match_db
 * @param func A user defined function for checking a match
 * @param args Propegated to "func"
 * @returns The number of invocations of "func"; i.e., the number of found 
 * candidates.
 */
int lnmu_iset_entry_validation(const struct lnmu_iset *iset,
                               const uint32_t *input,
                               int index,
                               lnmu_validation_lookup_func_t func,
                               void *args);

/**
 * @brief Perform classification using the RQ-RMI and isets of the latest
 * nuevomatch classifier wihin "lnm".
 * @param lnm Trainer object
 * @param header_fields Array of 5 32-bit values of LNMU_BATCH_SIZE packets.
 * @param priorities Array of size LNMU_BATCH_SIZE. Set per input packet.
 * @param ids Array of size LNMU_BATCH_SIZE. Set per input packet.
 * @param results Array of size LNMU_BATCH_SIZE. Set per packet. A pointer
 * to the user-defined "match" for each matching flow.
 * @param args Array of size LNMU_BATCH_SIZE. Set by this, a pointer to the 
 * user-defined "args" of per matching flow.
 * @return 0 on success. Use lnmu_get_error to see error message.
 */
int lnmu_classify(struct lnmu_trainer *lnm,
                  uint32_t *header_fields,
                  int *priorities,
                  uint32_t *ids,
                  void **results,
                  void **args);


/**
 * @brief Initiate range array data structure. Use this data structure to
 * manually store ranges and perform search / validation. Range arrays are not
 * part of iSets and so do not implement the "relaxed iSet" property as
 * described in the paper.
 * @param ranges Pointer to range array (each range is represented by its
 * beginning only).
 * @param size Size of the range array
 * @param compression Experimental feature. Reduce the size of the secondary
 * search array by introducing a linear validation phase per bucket. Use "1"
 * for regular behavior. Must be a power of two, positive integer.
 * @param after_train Set to true if the compression is performed after an
 * RQ-RMI model was trained over the ranges. Otherwise, set to false. This will
 * affect the validation phase complexity.
 * @returns A range array data structure
 */
struct lnmu_rangearr * lnmu_range_array_init(uint64_t *ranges,
                                             size_t size,
                                             int compression,
                                             int after_train);

/**
 * @brief Destroys a range array data structure.
 */
void lnmu_range_array_destroy(struct lnmu_rangearr *p);

/**
 * @brief Returns a pointer to the secondary search array of the range array
 * data structure.
 */
const uint64_t* lnmu_range_array_get_values(struct lnmu_rangearr *p);

/**
 * @brief Returns the size of the secondary search array of the range array
 * data structure.
 */
size_t lnmu_range_array_get_size(struct lnmu_rangearr *p);

/**
 * @brief Performs a secondary search within the range array, using batch size
 * of LNMU_BATCH_SIZE.
 * @param p A pointer to the range array data structure.
 * @param inputs Pointer to batch of inputs
 * @param rqrmi_outputs Pointer to a batch of RQRMI outputs
 * @param rqrmi_errors Pointer to a batch of RQRMI errors
 * @param found_values Pointer to a user-defined array that will be
 * modified by this with the found search values.
 * @param search_results Pointer to a user-defined array that will be modified
 * by this with the found positions.
 */
void lnmu_range_array_search_batch(struct lnmu_rangearr *p,
                                   const uint64_t *inputs,
                                   const double *rqrmi_outputs,
                                   const uint64_t *rqrmi_errors,
                                   uint64_t *found_values,
                                   int *search_results);


/**
 * @brief Performs the validation phase within the range array, using batch size
 * of LNMU_BATCH_SIZE.
 * @param p A pointer to the range array data structure.
 * @param inputs Pointer to batch of inputs
 * @param search_results A pointer to a batch of search results
 * @param found_values Pointer to a user-defined array that will be
 * modified by this with the found search values.
 * @param results Pointer to a user-defined array that will be modified
 * by this with the found positions.
 */
void lnmu_range_array_validate_batch(struct lnmu_rangearr *p,
                                     const uint64_t *inputs,
                                     const int *search_results,
                                     uint64_t *found_values,
                                     int *results);


/**
 * @brief Initiate the rqrmi64 data structure. Use for generating standalone
 * RQ-RMI models for 64bit inputs. These RQ-RMI models are not part of iSets
 * and should be use manually with the range array data structure.
 * @param config The RQ-RMI configuration. Use NULL for the default
 * configuration.
 * @param stage_widths An array of integers with the RQ-RMI widths. The fist
 * stage must have a width of 1. Use NULL for only one stage with one submodel.
 * @param stage_num Number of stages for the RQ-RMI model.
 */
struct lnmu_rqrmi64 *
lnmu_rqrmi64_init(struct lnmu_trainer_configuration *config,
                  const int *stage_widths,
                  size_t stage_num);

/**
 * @brief Destroy an rqrmi64 data structure.
 */
void lnmu_rqrmi64_destroy(struct lnmu_rqrmi64 *p);

/**
 * @brief Train the rqrmi64 data structure "p" over the array "values" of
 * size "num".
 * @returns 0 On success, otherwise 1.
 */
int lnmu_rqrmi64_train(struct lnmu_rqrmi64 *p,
                       const uint64_t *values,
                       size_t num);

/**
 * @brief Returns a pointer to the list of the RQ-RMI submodels' errors.
 * @param p A pointer to the rqrmi64 data structure.
 * @param size *size is set by this to hold the list size.
 * @returns A pointer to the error list. Negative values represent unreachable
 * submodels.
 */
const int * lnmu_rqrmi64_get_errors(struct lnmu_rqrmi64 *p, size_t *size);

/**
 * @brief Perform an inference over a batch of LNMU_BATCH_SIZE inputs.
 * @param p A pointer to the rqrmi64 data structure.
 * @param inputs A batch of inputs
 * @param rqrmi_outputs A pointer to a user-defined array that will be
 * modified by this with the rqrmi model outputs.
 * @param rqrmi_errors A pointer to a user-defined array that will be
 * modified by this with the rqrmi model errors.
 * @returns 0 if the model is valid. Otherwise returns 1.
 */
int lnmu_rqrmi64_inference_batch(struct lnmu_rqrmi64 *p,
                                 const uint64_t *inputs,
                                 double *rqrmi_outputs,
                                 uint64_t *rqrmi_errors);

/**
 * @brief Saves the RQ-RMI model in "p" to a memory location in "*ptr".
 * Sets "*size" to the size of the buffer. "p" must be initialized.
 * The user must manually free "*ptr" using libc "free" method.
 */
void lnmu_rqrmi64_store(struct lnmu_rqrmi64 *p, void **ptr, size_t *size);

/**
 * @brief Loads the RQ-RMI model in "p" from "ptr". "p" must be initialized,
 * and "ptr" must point to a buffer previously generated using
 * "lnmu_rqrmi64_store".
 */
void lnmu_rqrmi64_load(struct lnmu_rqrmi64 *p, void *ptr, size_t size);

#ifdef __cplusplus
}
#endif

#endif
